import { BasePrinter } from "@point_of_sale/app/utils/printer/base_printer";
import { getOnNotified } from "@point_of_sale/utils";
import { _t } from "@web/core/l10n/translation";
import { getTemplate } from "@web/core/templates";
import { createElement, append, createTextNode } from "@web/core/utils/xml";
import { uuid } from "@web/core/utils/strings";

const STATUS_ROLL_PAPER_HAS_RUN_OUT = 0x00080000;
const STATUS_ROLL_PAPER_HAS_ALMOST_RUN_OUT = 0x00020000;
const EPSON_ERRORS = {
    // Errors generated by Epson printer
    DeviceNotFound: _t(
        "Check the printer configuration for the 'Device ID' setting.\nIt should be set to: 'local_printer'"
    ),
    EPTR_AUTOMATICAL: _t(
        "Continuous printing of high-density printing caused a printing error. Please retry later"
    ),
    EPTR_COVER_OPEN: _t("Printer cover is open, please close it before printing"),
    EPTR_CUTTER: _t("The cutter has a foreign matter, please check the cutter mechanism"),
    EPTR_MECHANICAL: _t("Mechanical error, please check the printer"),
    EPTR_REC_EMPTY: _t("The paper is empty, please load paper into the printer"),
    EPTR_UNRECOVERABLE: _t("Low voltage unrecoverable error occured, please check the printer"),
    EX_BADPORT: _t("The device is not connected, please check the printer power / connection"),
    EX_TIMEOUT: _t("Timeout occured, please try again"),
    // Odoo Errors
    ERROR_CODE_PRINTER_NOT_REACHABLE: _t("The printer is not reachable."),
    ERROR_CODE_REQUEST_BACKEND: _t(
        "Failed to add receipt to print queue. Please check your internet connection and try again."
    ),
    ERROR_CODE_PRINTER_RESPONSE_TIMEOUT: _t(
        "Printer took too long to respond with the printing result. Please check the printer and try again. Ensure that the printer is online and a url/id are set in its Server Direct Print configuration\n"
    ),
    ERROR_CODE_PRINTER_OLD_FIRMWARE: _t(
        "The printer firmware is too old, please update it on the printer settings page before using it in Point Of Sale."
    ),
};

/**
 * Sends print request to ePos printer that is directly connected to the local network.
 */
export class EpsonPrinter extends BasePrinter {
    setup({ ip }) {
        super.setup(...arguments);
        this.url = window.location.protocol + "//" + ip;
        this.address = this.url + "/cgi-bin/epos/service.cgi?devid=local_printer";
        this.layout_xml = "ePOSLayout";
    }

    epsonPrint(children) {
        let layout = getTemplate(`point_of_sale.${this.layout_xml}`);
        if (!layout) {
            throw new Error(`'${this.layout_xml}' not loaded`);
        }
        layout = layout.cloneNode(true);
        // Assign a job id to the print job
        if (this.layout_xml === "ServerDirectPrintLayout") {
            const [eposJobIdEl] = layout.getElementsByTagName("printjobid");
            // The print job ID must have 1 to 30 characters and allowable characters are
            // alphanumerical characters, underscore, hyphen and period.
            this.printJobId = uuid();
            append(eposJobIdEl, this.printJobId);
        }
        const [eposPrintEl] = layout.getElementsByTagName("epos-print");
        append(eposPrintEl, children);
        // IMPORTANT: Need to remove `xmlns=""` in the image and cut elements.
        // > Otherwise, the print request will succeed but it the printer device won't actually do the printing.
        return layout.innerHTML.replaceAll(`xmlns=""`, "");
    }

    /**
     * @override
     * Create the raster data from a canvas
     */
    processCanvas(canvas) {
        const rasterData = this.canvasToRaster(canvas);
        const encodedData = this.encodeRaster(rasterData);
        return this.epsonPrint([
            createElement(
                "image",
                {
                    width: canvas.width,
                    height: canvas.height,
                    align: "center",
                },
                [createTextNode(encodedData)]
            ),
            createElement("cut", { type: "feed" }),
        ]);
    }

    /**
     * @override
     */
    openCashbox() {
        const pulse = this.epsonPrint([createElement("pulse")]);
        this.sendPrintingJob(pulse);
    }

    /*
     * This method sends a print job to the Epson ePoS printer via a fetch request
     * and waits for a notification via WebSocket to confirm the print job status.
     * It returns an object indicating the result of the print job.
     * @param {String} img - The receipt image data to be printed.
     * @returns {Object} - An object containing the result of the print job, error code (if any), and a flag indicating if retry is possible.
     */
    async sendPrintingJob(img) {
        try {
            const res = await fetch(this.address, {
                method: "POST",
                body: img,
            });
            const body = await res.text();
            const parser = new DOMParser();
            const parsedBody = parser.parseFromString(body, "application/xml");
            const response = parsedBody.querySelector("response");
            return {
                result: response.getAttribute("success") === "true",
                errorCode: response.getAttribute("code"),
                status: parseInt(response.getAttribute("status")) || 0,
                canRetry: true,
            };
        } catch {
            return {
                result: false,
                canRetry: true,
                errorCode: "ERROR_CODE_PRINTER_NOT_REACHABLE",
            };
        }
    }

    /**
     * Transform a (potentially colored) canvas into a monochrome raster image.
     * We will use Floyd-Steinberg dithering.
     */
    canvasToRaster(canvas) {
        const imageData = canvas.getContext("2d").getImageData(0, 0, canvas.width, canvas.height);
        const pixels = imageData.data;
        const width = imageData.width;
        const height = imageData.height;
        const errors = Array.from(Array(width), (_) => Array(height).fill(0));
        const rasterData = new Array(width * height).fill(0);

        for (let y = 0; y < height; y++) {
            for (let x = 0; x < width; x++) {
                let oldColor, newColor;

                // Compute grayscale level. Those coefficients were found online
                // as R, G and B have different impacts on the darkness
                // perception (e.g. pure blue is darker than red or green).
                const idx = (y * width + x) * 4;
                oldColor = pixels[idx] * 0.299 + pixels[idx + 1] * 0.587 + pixels[idx + 2] * 0.114;

                // Propagate the error from neighbor pixels
                oldColor += errors[x][y];
                oldColor = Math.min(255, Math.max(0, oldColor));

                if (oldColor < 128) {
                    // This pixel should be black
                    newColor = 0;
                    rasterData[y * width + x] = 1;
                } else {
                    // This pixel should be white
                    newColor = 255;
                    rasterData[y * width + x] = 0;
                }

                // Propagate the error to the following pixels, based on
                // Floyd-Steinberg dithering.
                const error = oldColor - newColor;
                if (error) {
                    if (x < width - 1) {
                        // Pixel on the right
                        errors[x + 1][y] += (7 / 16) * error;
                    }
                    if (x > 0 && y < height - 1) {
                        // Pixel on the bottom left
                        errors[x - 1][y + 1] += (3 / 16) * error;
                    }
                    if (y < height - 1) {
                        // Pixel below
                        errors[x][y + 1] += (5 / 16) * error;
                    }
                    if (x < width - 1 && y < height - 1) {
                        // Pixel on the bottom right
                        errors[x + 1][y + 1] += (1 / 16) * error;
                    }
                }
            }
        }

        return rasterData.join("");
    }

    /**
     * Base 64 encode a raster image
     */
    encodeRaster(rasterData) {
        let encodedData = "";
        for (let i = 0; i < rasterData.length; i += 8) {
            const sub = rasterData.substr(i, 8);
            encodedData += String.fromCharCode(parseInt(sub, 2));
        }
        return btoa(encodedData);
    }

    /**
     * @override
     */
    getActionError() {
        const printRes = super.getResultsError();
        if (window.location.protocol === "https:") {
            printRes.message.body += _t(
                "If you are on a secure server (HTTPS) please make sure you manually accepted the certificate by accessing %s. ",
                this.url
            );
        }
        return printRes;
    }

    hasStatus(status, attribute) {
        //The status is a bit field
        return (status & attribute) === attribute;
    }

    /**
     * @override
     */
    getResultsError(printResult) {
        const errorCode = printResult.errorCode;
        const status = printResult.status;
        let message = _t(
            "The following error code was given by the printer: %s \nTo find more details on the error reason, please search online for: Epson Server Direct Print %s ",
            errorCode,
            errorCode
        );
        if (this.hasStatus(status, STATUS_ROLL_PAPER_HAS_RUN_OUT)) {
            message = EPSON_ERRORS["EPTR_REC_EMPTY"];
        } else {
            message = EPSON_ERRORS[errorCode] || message;
        }
        // https://download4.epson.biz/sec_pubs/pos/reference_en/epos_print/ref_epos_print_xml_en_xmlforcontrollingprinter_response.html
        if (this.printJobId) {
            message += _t("\nPrint Job ID: %s", this.printJobId);
        }
        return {
            successful: false,
            errorCode: errorCode,
            status: status,
            message: {
                title: _t("Printing failed"),
                body: message,
            },
            canRetry: printResult.canRetry || false,
        };
    }

    getResultWarningCode(printResult) {
        const status = printResult?.status;
        if (!status) {
            return undefined;
        }
        if (this.hasStatus(status, STATUS_ROLL_PAPER_HAS_ALMOST_RUN_OUT)) {
            return "ROLL_PAPER_HAS_ALMOST_RUN_OUT";
        }
        return undefined;
    }
}

export class EpsonServerDirectPrinter extends EpsonPrinter {
    // Documentation: https://files.support.epson.com/pdf/pos/bulk/tm-int_sdp_um_e_reve.pdf
    setup({ posConfigId, posData, busService }) {
        super.setup(...arguments);
        this.layout_xml = "ServerDirectPrintLayout";
        this.posData = posData;
        this.posConfigId = posConfigId;
        const websocketChannel = "EPSON_SERVER_DIRECT_PRINT";
        getOnNotified(busService, websocketChannel);
        posData.connectWebSocket(
            websocketChannel,
            this.epsonServerDirectPrintNotification.bind(this)
        );
    }

    /*
     * This method is called when the Epson Server Direct Print printer
     * polls the controller with a print job result, then sends a notification via WebSocket.
     * It resolves the pending print job promise with the received data.
     * If the printer firmware is old and doesn't support print job IDs,
     * it sets an appropriate error code.
     */
    epsonServerDirectPrintNotification(printerResponse) {
        let jobResult = printerResponse.find((result) => result.printJobId === this.printJobId);
        if (jobResult) {
            printerResponse.success = jobResult.success; // "true" or "false" as string
            printerResponse.errorCode = jobResult.errorCode;
        } else if (printerResponse.some((result) => !result.printJobId)) {
            // Old firmware doesn't handle job id usage, notifications won't work until firmware is updated
            jobResult = {
                success: "false",
                errorCode: "ERROR_CODE_PRINTER_OLD_FIRMWARE",
            };
        }
        if (jobResult) {
            this.pendingPrintJobResolver?.(jobResult); // Resolves the printerResponsePromise
            this.pendingPrintJobResolver = null;
        }
    }

    /*
     * This method sends a print job to the Epson Server Direct Print printer via a backend controller
     * and waits for a notification via WebSocket to confirm the print job status.
     * It returns an object indicating the result of the print job.
     * @param {String} img - The receipt image data to be printed.
     * @returns {Object} - An object containing the result of the print job, error code (if any), and a flag indicating if retry is possible.
     */
    async sendPrintingJob(img) {
        // This promise will be resolved when the controller sends a notification via WebSocket
        const printerResponsePromise = new Promise((resolve) => {
            this.pendingPrintJobResolver = resolve;
            // Automatically resolve with an error if timeout
            setTimeout(() => {
                resolve({ result: false, errorCode: "ERROR_CODE_PRINTER_RESPONSE_TIMEOUT" });
            }, 10000);
        });
        try {
            await this.posData.orm.call("pos.config", "add_receipt_to_print_queue", [
                this.posConfigId,
                img,
            ]);
            const printerResponse = await printerResponsePromise;
            return {
                result: printerResponse.success === "true" && !printerResponse.errorCode,
                errorCode: printerResponse.errorCode,
                canRetry: true,
            };
        } catch {
            // Failed to contact backend to add receipt to print queue
            this.pendingPrintJobResolver = null;
            return {
                result: false,
                errorCode: "ERROR_CODE_REQUEST_BACKEND",
                canRetry: true,
            };
        }
    }
}
