/** @odoo-module */

import { migrate, ODOO_VERSION } from "@spreadsheet/o_spreadsheet/migration";
import { Model } from "@odoo/o-spreadsheet";

QUnit.module("spreadsheet > migrations");

QUnit.test("Odoo formulas are migrated", (assert) => {
    const data = {
        sheets: [
            {
                cells: {
                    A1: { content: `=PIVOT("1")` },
                    A2: { content: `=PIVOT.HEADER("1")` },
                    A3: { content: `=FILTER.VALUE("1")` },
                    A4: { content: `=LIST("1")` },
                    A5: { content: `=LIST.HEADER("1")` },
                    A6: { content: `=PIVOT.POSITION("1")` },
                    A7: { content: `=pivot("1")` },
                },
            },
        ],
    };
    const migratedData = migrate(data);
    assert.strictEqual(migratedData.sheets[0].cells.A1.content, `=PIVOT.VALUE("1")`);
    assert.strictEqual(migratedData.sheets[0].cells.A2.content, `=PIVOT.HEADER("1")`);
    assert.strictEqual(migratedData.sheets[0].cells.A3.content, `=ODOO.FILTER.VALUE("1")`);
    assert.strictEqual(migratedData.sheets[0].cells.A4.content, `=ODOO.LIST("1")`);
    assert.strictEqual(migratedData.sheets[0].cells.A5.content, `=ODOO.LIST.HEADER("1")`);
    assert.strictEqual(migratedData.sheets[0].cells.A6.content, `=ODOO.PIVOT.POSITION("1")`);
    assert.strictEqual(migratedData.sheets[0].cells.A7.content, `=PIVOT.VALUE("1")`);
});

QUnit.test("Pivot 'day' arguments are migrated", (assert) => {
    const data = {
        odooVersion: 1,
        sheets: [
            {
                cells: {
                    A1: { content: `=ODOO.PIVOT("1","21/07/2022")` },
                    A2: { content: `=ODOO.PIVOT.HEADER("1","11/12/2022")` },
                    A3: { content: `=odoo.pivot("1","21/07/2021")` },
                    A4: { content: `=ODOO.PIVOT("1","test")` },
                    A5: { content: `=odoo.pivot("1","21/07/2021")+"21/07/2021"` },
                    A6: { content: `=BAD_FORMULA(` },
                },
            },
        ],
    };
    const migratedData = migrate(data);
    assert.strictEqual(migratedData.sheets[0].cells.A1.content, `=PIVOT.VALUE("1","07/21/2022")`);
    assert.strictEqual(migratedData.sheets[0].cells.A2.content, `=PIVOT.HEADER("1","12/11/2022")`);
    assert.strictEqual(migratedData.sheets[0].cells.A3.content, `=PIVOT.VALUE("1","07/21/2021")`);
    assert.strictEqual(migratedData.sheets[0].cells.A4.content, `=PIVOT.VALUE("1","test")`);
    assert.strictEqual(
        migratedData.sheets[0].cells.A5.content,
        `=PIVOT.VALUE("1","07/21/2021")+"21/07/2021"`
    );
    assert.strictEqual(migratedData.sheets[0].cells.A6.content, `=BAD_FORMULA(`);
});

QUnit.test("Global filters: pivot fields is correctly added", (assert) => {
    const data = {
        globalFilters: [
            {
                id: "Filter1",
                type: "relation",
                label: "Relation Filter",
                fields: {
                    1: {
                        field: "foo",
                        type: "char",
                    },
                },
            },
        ],
        pivots: {
            1: {
                name: "test",
                measures: [],
                colGroupBys: [],
                rowGroupBys: [],
            },
        },
    };
    const migratedData = migrate(data);
    const filter = migratedData.globalFilters[0];
    const pivot = migratedData.pivots["1"];
    assert.deepEqual(pivot.fieldMatching, {
        Filter1: {
            chain: "foo",
            type: "char",
        },
    });
    assert.strictEqual(filter.fields, undefined);
});

QUnit.test("Global filters: date is correctly migrated", (assert) => {
    const data = {
        globalFilters: [
            {
                id: "1",
                type: "date",
                rangeType: "year",
                defaultValue: { year: "last_year" },
            },
            {
                id: "2",
                type: "date",
                rangeType: "year",
                defaultValue: { year: "antepenultimate_year" },
            },
            {
                id: "3",
                type: "date",
                rangeType: "year",
                defaultValue: { year: "this_year" },
            },
        ],
    };
    const migratedData = migrate(data);
    const [f1, f2, f3] = migratedData.globalFilters;
    assert.deepEqual(f1.defaultValue, { yearOffset: -1 });
    assert.deepEqual(f2.defaultValue, { yearOffset: -2 });
    assert.deepEqual(f3.defaultValue, { yearOffset: 0 });
});

QUnit.test("List name default is model name", (assert) => {
    const data = {
        lists: {
            1: {
                name: "Name",
                model: "Model",
            },
            2: {
                model: "Model",
            },
        },
    };
    const migratedData = migrate(data);
    assert.strictEqual(Object.values(migratedData.lists).length, 2);
    assert.strictEqual(migratedData.lists["1"].name, "Name");
    assert.strictEqual(migratedData.lists["2"].name, "Model");
});

QUnit.test("Pivot name default is model name", (assert) => {
    const data = {
        pivots: {
            1: {
                name: "Name",
                model: "Model",
                measures: [],
                colGroupBys: [],
                rowGroupBys: [],
            },
            2: {
                model: "Model",
                measures: [],
                colGroupBys: [],
                rowGroupBys: [],
            },
        },
    };
    const migratedData = migrate(data);
    assert.strictEqual(Object.values(migratedData.pivots).length, 2);
    assert.strictEqual(migratedData.pivots["1"].name, "Name");
    assert.strictEqual(migratedData.pivots["2"].name, "Model");
});

QUnit.test("fieldMatchings are moved from filters to their respective datasources", (assert) => {
    const data = {
        globalFilters: [
            {
                id: "Filter",
                label: "MyFilter1",
                type: "relation",
                listFields: {
                    1: {
                        field: "parent_id",
                        type: "many2one",
                    },
                },
                pivotFields: {
                    1: {
                        field: "parent_id",
                        type: "many2one",
                    },
                },
                graphFields: {
                    fig1: {
                        field: "parent_id",
                        type: "many2one",
                    },
                },
            },
        ],
        pivots: {
            1: {
                name: "Name",
                measures: [],
                colGroupBys: [],
                rowGroupBys: [],
            },
        },
        lists: {
            1: {
                name: "Name",
            },
        },
        sheets: [
            {
                figures: [
                    {
                        id: "fig1",
                        tag: "chart",
                        data: {
                            type: "odoo_bar",
                        },
                    },
                ],
            },
        ],
    };
    const migratedData = migrate(data);
    assert.deepEqual(migratedData.pivots["1"].fieldMatching, {
        Filter: { chain: "parent_id", type: "many2one" },
    });
    assert.deepEqual(migratedData.lists["1"].fieldMatching, {
        Filter: { chain: "parent_id", type: "many2one" },
    });
    assert.deepEqual(migratedData.sheets[0].figures[0].data.fieldMatching, {
        Filter: { chain: "parent_id", type: "many2one" },
    });
});

QUnit.test("fieldMatchings offsets are correctly preserved after migration", (assert) => {
    const data = {
        globalFilters: [
            {
                id: "Filter",
                label: "MyFilter1",
                type: "relation",
                listFields: {
                    1: {
                        field: "parent_id",
                        type: "date",
                        offset: "-1",
                    },
                },
                pivotFields: {
                    1: {
                        field: "parent_id",
                        type: "date",
                        offset: "-1",
                    },
                },
                graphFields: {
                    fig1: {
                        field: "parent_id",
                        type: "date",
                        offset: "-1",
                    },
                },
            },
        ],
        pivots: {
            1: {
                name: "Name",
                measures: [],
                colGroupBys: [],
                rowGroupBys: [],
            },
        },
        lists: {
            1: {
                name: "Name",
            },
        },
        sheets: [
            {
                figures: [
                    {
                        id: "fig1",
                        tag: "chart",
                        data: {
                            type: "odoo_bar",
                        },
                    },
                ],
            },
        ],
    };
    const migratedData = migrate(data);
    assert.deepEqual(migratedData.pivots["1"].fieldMatching, {
        Filter: { chain: "parent_id", type: "date", offset: "-1" },
    });
    assert.deepEqual(migratedData.lists["1"].fieldMatching, {
        Filter: { chain: "parent_id", type: "date", offset: "-1" },
    });
    assert.deepEqual(migratedData.sheets[0].figures[0].data.fieldMatching, {
        Filter: { chain: "parent_id", type: "date", offset: "-1" },
    });
});

QUnit.test("group year/quarter/month filters to a single filter type", (assert) => {
    const data = {
        version: 14,
        odooVersion: 5,
        globalFilters: [
            {
                id: "1",
                type: "relation",
                label: "a relational filter",
                defaultValue: [2],
                defaultValueDisplayNames: ["Mitchell Admin"],
                modelName: "res.users",
            },
            {
                id: "2",
                type: "date",
                label: "a year relational filter",
                rangeType: "year",
                defaultsToCurrentPeriod: true,
            },
            {
                id: "3",
                type: "date",
                label: "a quarter relational filter",
                rangeType: "quarter",
                defaultsToCurrentPeriod: true,
            },
            {
                id: "4",
                type: "date",
                label: "a month relational filter",
                rangeType: "month",
                defaultsToCurrentPeriod: true,
            },
            {
                id: "5",
                type: "date",
                label: "a relative date filter",
                defaultValue: "last_week",
                rangeType: "relative",
                defaultsToCurrentPeriod: false,
            },
        ],
    };
    const migratedData = migrate(data);
    const filters = migratedData.globalFilters;
    assert.deepEqual(filters, [
        {
            id: "1",
            type: "relation",
            label: "a relational filter",
            defaultValue: [2],
            defaultValueDisplayNames: ["Mitchell Admin"],
            modelName: "res.users",
        },
        {
            id: "2",
            type: "date",
            label: "a year relational filter",
            rangeType: "fixedPeriod",
            defaultValue: "this_year",
        },
        {
            id: "3",
            type: "date",
            label: "a quarter relational filter",
            rangeType: "fixedPeriod",
            defaultValue: "this_quarter",
        },
        {
            id: "4",
            type: "date",
            label: "a month relational filter",
            rangeType: "fixedPeriod",
            defaultValue: "this_month",
        },
        {
            id: "5",
            type: "date",
            label: "a relative date filter",
            rangeType: "relative",
            defaultValue: "last_week",
        },
    ]);
});

QUnit.test("Pivot are migrated from 6 to 9", (assert) => {
    const data = {
        pivots: {
            1: {
                name: "Name",
                model: "Model",
                measures: [],
                colGroupBys: [],
                rowGroupBys: [],
                fieldMatching: { 1: { chain: "foo", type: "char" } },
            },
        },
    };
    const migratedData = migrate(data);
    assert.strictEqual(Object.values(migratedData.pivots).length, 1);
    assert.deepEqual(migratedData.pivots["1"], {
        type: "ODOO",
        fieldMatching: { 1: { chain: "foo", type: "char" } },
        name: "Name",
        model: "Model",
        measures: [],
        columns: [],
        rows: [],
        formulaId: "1",
    });
});

QUnit.test("Pivot are migrated from 9 to 10", (assert) => {
    const data = {
        odooVersion: 9,
        pivots: {
            1: {
                type: "ODOO",
                name: "Name",
                model: "res.model",
                measures: ["probability"],
                colGroupBys: ["foo"],
                rowGroupBys: ["create_date:month"],
                formulaId: "1",
            },
        },
    };
    const migratedData = migrate(data);
    assert.strictEqual(Object.values(migratedData.pivots).length, 1);
    assert.deepEqual(migratedData.pivots["1"], {
        type: "ODOO",
        name: "Name",
        model: "res.model",
        measures: [{ name: "probability" }],
        columns: [{ name: "foo" }],
        rows: [{ name: "create_date", granularity: "month" }],
        formulaId: "1",
    });
});

QUnit.test("Pivot formulas are migrated from 9 to 10", (assert) => {
    const data = {
        odooVersion: 9,
        sheets: [
            {
                cells: {
                    A1: { content: `=ODOO.PIVOT("1")` },
                    A2: { content: `=ODOO.PIVOT.HEADER("1")` },
                    A3: { content: `=ODOO.PIVOT.POSITION("1")` },
                    A4: { content: `=ODOO.PIVOT.TABLE("1")` },
                    A5: { content: `=odoo.pivot("1")` },
                },
            },
        ],
    };
    const migratedData = migrate(data);
    assert.strictEqual(migratedData.sheets[0].cells.A1.content, `=PIVOT.VALUE("1")`);
    assert.strictEqual(migratedData.sheets[0].cells.A2.content, `=PIVOT.HEADER("1")`);
    assert.strictEqual(migratedData.sheets[0].cells.A3.content, `=ODOO.PIVOT.POSITION("1")`);
    assert.strictEqual(migratedData.sheets[0].cells.A4.content, `=PIVOT("1")`);
    assert.strictEqual(migratedData.sheets[0].cells.A5.content, `=PIVOT.VALUE("1")`);
});

QUnit.test("Pivot formulas using pivot positions are migrated (11 to 12)", (assert) => {
    const data = {
        odooVersion: 9,
        sheets: [
            {
                cells: {
                    A1: { content: `=-PIVOT.VALUE("1","balance","account_id",ODOO.PIVOT.POSITION("1","account_id",12),"date:quarter","4/"&ODOO.FILTER.VALUE("Year"))` },
                    A2: { content: `=PIVOT.HEADER("1","account_id",ODOO.PIVOT.POSITION("1","account_id",14))` },
                    A3: { content: `=ODOO.PIVOT.POSITION("1","account_id",14)` },
                    A4: { content: `=ODOO.PIVOT.POSITION("1",14)` },
                },
            },
        ],
    };
    const migratedData = migrate(data);
    assert.strictEqual(migratedData.sheets[0].cells.A1.content, `=-PIVOT.VALUE("1","balance","#account_id",12,"date:quarter","4/"&ODOO.FILTER.VALUE("Year"))`);
    assert.strictEqual(migratedData.sheets[0].cells.A2.content, `=PIVOT.HEADER("1","#account_id",14)`);
    assert.strictEqual(migratedData.sheets[0].cells.A3.content, `=ODOO.PIVOT.POSITION("1","account_id",14)`);
    assert.strictEqual(migratedData.sheets[0].cells.A4.content,  `=ODOO.PIVOT.POSITION("1",14)`);
});

QUnit.test("Odoo version is exported", (assert) => {
    const model = new Model();
    assert.strictEqual(model.exportData().odooVersion, ODOO_VERSION);
});
